<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\Log;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;

class CategoryController extends Controller
{
    // Fetch all categories
    

    public function index(Request $request)
    {
        $companyId = getCompanyId();
        $query = Category::where('company_id', $companyId)
                    ->orderBy('created_at', 'desc');

        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where('name', 'like', '%' . $search . '%');
        }
        $categories = $query->paginate(10);
        $categories->transform(function ($item) {
            $item->hashed_id = substr(md5($item->id), 0, 8);
            return $item;
        });
        return response()->json($categories);

    }
    
    
    

    


    // Show a specific category
    public function show()
    {
        $companyId = getCompanyId();

        $categories = Category::where('company_id', $companyId)
            ->orderBy('created_at', 'desc') 
            ->get();
        return response()->json($categories);
    }

    

    public function store(Request $request)
    {
        try {
            $companyId = null;  
            $companyId = getCompanyId();
    
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable',
            ]);
    
            // Create the category with the logged-in company's ID
            $category = Category::create([
                'name' => $validated['name'],
                'description' => $validated['description'],
                'company_id' => $companyId,
            ]);
    
            return response()->json(['category' => $category], 201);
        } catch (\Exception $e) {
            Log::error("Error creating category: " . $e->getMessage());
            return response()->json(['error' => 'Failed to create category'], 500);
        }
    }
    
    
    

    // Update an existing category
    public function update(Request $request, Category $category)
    {
        try {
            // Determine the company ID based on the authenticated user or admin
            $companyId = null;
    
            $companyId = getCompanyId();
    
            // Check if the category belongs to the logged-in company
            if ($category->company_id !== $companyId) {
                return response()->json(['error' => 'Forbidden'], 403);
            }
    
            // Validate the request data
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable',
            ]);
    
            // Update the category with the validated data
            $category->update($validated);
    
            $category->hashed_id = substr(md5($category->id), 0, 8);
            // Return the updated category
            return response()->json($category);
        } catch (\Exception $e) {
            Log::error("Error updating category: " . $e->getMessage());
            return response()->json(['error' => 'Failed to update category'], 500);
        }
    }
    

    // Delete a category
    public function destroy(Category $category)
    {
        $category->delete(); // Delete the category

        // Return a 204 No Content response
        return response()->noContent();
    }


    public function CategoryCount()
    {
        try {
            $companyId = getCompanyId();

            // Count only the CompanyUser records with the matching company_id
            $count = Category::where('company_id', $companyId)->count();
    
            return response()->json(['count' => $count]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Unable to fetch company user count'], 500);
        }
    }

}



