<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\CompanyProfile;
use Illuminate\Support\Facades\Auth;
use App\Models\Company;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;


class CompanyProfileController extends Controller
{
    /**
     * Display the company profile for the authenticated admin.
     */
    public function index()
    {
        $companyId = getCompanyId();
        
        $company = CompanyProfile::where('company_id', $companyId)->first();

        if (!$company) {
            return response()->json(['message' => 'Company not found'], 404);
        }
        return response()->json($company);
    }

    /**
     * Store or update the company profile.
     */
    public function store(Request $request)
    {
        try {
            $company = auth('admin')->user();
            $companyId = $company->id;
    
            // Validate the incoming data
            $validated = $request->validate([
                'name' => 'required|string',
                'address' => 'required|string',
                'phone_number' => 'required|string',
                'email' => 'required|email',
                'website' => 'nullable|string',
                'about' => 'nullable|string',
                'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:1024', // Validate the logo image
            ]);
    
            // Handle file upload for logo
            $logoPath = null;
            if ($request->hasFile('logo')) {
                $extension = $request->file('logo')->getClientOriginalExtension();
                $logoName = 'logo_' . $companyId . '_' . time() . '.' . $extension;
                $logoPath = $request->file('logo')->storeAs('company_logos', $logoName, 'public');
            }
    
            // Create a new company profile
            $companyProfile = CompanyProfile::create([
                'name' => $validated['name'],
                'address' => $validated['address'],
                'phone_number' => $validated['phone_number'],
                'email' => $validated['email'],
                'website' => $validated['website'],
                'about' => $validated['about'],
                'logo' => $logoPath,
                'company_id' => $companyId,
            ]);
    
            // return response()->json($companyProfile, 200);
            return response()->json($companyProfile, 200);
    
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred while saving the profile.'], 500);
        }
    }

/**
 * Update the specified company profile.
 */
public function update(Request $request, $id)
{
    try {
        $company = auth('admin')->user();
        $companyId = $company->id;


        $companyProfile = CompanyProfile::where('company_id', $companyId)->findOrFail($id);
        // Validate the incoming data
        $validated = $request->validate([
            'name' => 'required|string',
            'address' => 'required|string',
            'phone_number' => 'required|string',
            'email' => 'required|email',
            'website' => 'nullable|string',
            'about' => 'nullable|string',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:1024', // Validate the logo image
        ]);

        // Handle file upload for logo
        $logoPath = null;
        if ($request->hasFile('logo')) {
            $logoFile = $request->file('logo');
            $extension = $logoFile->getClientOriginalExtension();
            $logoName = 'logo_' . $companyId . '_' . time() . '.' . $extension;
            $logoPath = $logoFile->storeAs('company_logos', $logoName, 'public');


            if($companyProfile->logo){
                Storage::disk('public')->delete($companyProfile->logo);
            }

            $validated['logo'] = $logoPath;
        }

        // Create a new company profile
        $companyProfile->update($validated);

        // return response()->json($companyProfile, 200);
        return response()->json($companyProfile, 200);

    } catch (\Exception $e) {
        return response()->json(['error' => 'An error occurred while saving the profile.'], 500);
    }
}







    
    
    

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        // Deleting logic if needed
    }

    public function TopCompanies()
    {
        // Base query
        Log::info('welcome to top company');
        $query = Company::with('profile')
            ->withCount('users')
            ->withCount(['product', 'saleItem as sales_count' => function($q) {
                // count sales by company
                $q->select(DB::raw('COUNT(*)'));
            }]);

        // Order by users count descending (change to 'products_count' or 'sales_count' if you want)
        $query->orderByDesc('users_count');
        $companies = $query->take(10)->get();
        Log::info(['top companies:'=> $companies]);

        return response()->json($companies);
    }

}
