<?php

namespace App\Http\Controllers;

use App\Models\CompanyUser;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;


class CompanyUserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $companyId = getCompanyId();

        $query = CompanyUser::where('company_id', $companyId)
                    ->orderBy('created_at', 'desc');

        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', '%' . $search . '%')
                ->orWhere('email', 'like', '%' . $search . '%');
            });
        }

        $companyUsers = $query->paginate(8);
        $companyUsers->transform(function ($item) {
            $item->hashed_id = substr(md5($item->id), 0, 8);
            return $item;
        });

        return response()->json($companyUsers);
    }
    

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(CompanyUser $companyUser)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(CompanyUser $companyUser)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $user = CompanyUser::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:company_users,email,' . $id,
            'password' => 'nullable|string|min:6',
            'is_locked' => 'required|boolean',
        ]);

        $data = [
            'name' => $request->name,
            'email' => $request->email,
            'is_locked' => $request->is_locked,
        ];

        if ($request->filled('password')) {
            $data['password'] = bcrypt($request->password);
        }

        $user->update($data);

        return response()->json(['message' => 'User updated successfully']);
    }
    
    
    
    public function destroy(CompanyUser $companyUser)
    {
        try {
            if ($companyUser->profile_image) {
                Storage::delete('public/' . $companyUser->profile_image);
            }
            $companyUser->delete();
            return response()->json(['message' => 'Company user deleted successfully'], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to delete company user'], 500);
        }
    }
    

    //======================================


    public function CompanyUserCount()
    {
        try {
            $companyId = getCompanyId(); // Use the authenticated company's ID
            $count = CompanyUser::where('company_id', $companyId)->count();
            return response()->json(['count' => $count]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Unable to fetch company user count'], 500);
        }
    }


    //====================

    public function getActiveUserCount()
    {
            $companyId = getCompanyId(); // Use the authenticated company's ID
            $activeUsers = CompanyUser::where('company_id', $companyId)
            ->where('last_seen', '>=' , Carbon::now()->subMinutes(5))
            ->count();

            return response()->json(['active_users' => $activeUsers]);
    }



    //===========================

    public function getActiveUsers(){
        $companyId = getCompanyId(); // Use the authenticated company's ID
            $activeUsers = CompanyUser::where('company_id', $companyId)
            ->where('last_seen', '>=' , Carbon::now()->subMinutes(5))
            ->get();

            return response()->json($activeUsers);
    }

    public function getUsersList()
    {
        $companyId = getCompanyId();
        $users = CompanyUser::where('company_id', $companyId)
        ->select('name', 'email', 'profile_image' , 'phone_number', 'register_date', 'address', 'last_seen')
        ->orderBy('last_seen', 'desc')
            ->get()
            ->map(function ($user) {
                return [
                    'name' => $user->name,
                    'email' => $user->email,
                    'profile_image' => $user->profile_image,
                    'phone_number' => $user->phone_number,
                    'register_date' => $user->register_date,
                    'address' => $user->address,
                    'last_seen' => $user->last_seen ? Carbon::parse($user->last_seen)->diffForHumans() : 'Never',
                ];
            });

        return response()->json($users);
    }



    //==========================

    public function getUserProfile()
    {
        $userId = auth('user')->user()->id;
        $companyId = getCompanyId();
        $companyUsers = CompanyUser::where('id', $userId)
        ->where('company_id', $companyId)
        ->first();
        return response()->json($companyUsers);
    }


    //===========================

    public function updateUserProfile(Request $request)
    {
        $user = auth('user')->user();
        $companyUser = CompanyUser::find($user->id);

        if (!$companyUser) {
            return response()->json(['message' => 'User not found'], 404);
        }

        // Update text fields
        $companyUser->name = $request->name;
        $companyUser->email = $request->email;
        $companyUser->phone_number = $request->phone_number;
        $companyUser->address = $request->address;

        // Handle profile image
        if ($request->hasFile('profile_image')) {
            $oldImage = $companyUser->profile_image;
            if ($oldImage && Storage::exists('public/' . $oldImage)) {
                Storage::delete('public/' . $oldImage);
            }
        
            $newImage = $request->file('profile_image')->store('profiles', 'public');
            $companyUser->profile_image = $newImage;
        }

        $companyUser->save();
        return response()->json(['message' => 'Profile updated successfully!', 'user' => $companyUser]);
    }
}
