<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\Log;

use App\Models\MedicineType;
use Illuminate\Http\Request;

class MedicineTypeController extends Controller {
    public function index(Request $request) {
        $companyId = getCompanyId();
        $query = MedicineType::where('company_id', $companyId)
                    ->orderBy('created_at', 'desc');

        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where('name', 'like', '%' . $search . '%');
            $query->orWhere('abbreviation', 'like', '%' . $search . '%');
        }
        $medicineType = $query->paginate(10);
        $medicineType->transform(function ($item) {
            $item->hashed_id = substr(md5($item->id), 0, 8);
            return $item;
        });

        return response()->json($medicineType);
    
    }

    public function store(Request $request) {

        $companyId = getCompanyId();

        $validated = $request->validate([
            'name' => 'required|unique:medicine_types,name',
            'abbreviation' => 'required|unique:medicine_types,abbreviation|max:10',
            'description' => 'nullable',
        ]);

        $medicine_type = MedicineType::create([
            'name' => $validated['name'],
            'abbreviation' => $validated['abbreviation'],
            'description' => $validated['description'],
            'company_id' => $companyId,
        ]);

        return response()->json(['medicine_type' => $medicine_type], 201);
    }

    public function show() {
        $companyId = getCompanyId();
        $medicineType = MedicineType::where('company_id', $companyId)
        ->orderBy('created_at', 'desc')
        ->get();

        return response()->json($medicineType);
    }

    public function update(Request $request, $id) {

        Log::info('Welcome to update method of medicine type');

        $medicineType = MedicineType::findOrFail($id);

        Log::info(['type'=> $id]);

        $validated = $request->validate([
            'name' => 'required|unique:medicine_types,name,' . $id,
            'abbreviation' => 'required|unique:medicine_types,abbreviation,' . $id . '|max:10',
            'description' => 'nullable',
        ]);

        $medicineType->update($validated);

        return $medicineType;
    }

    public function destroy(MedicineType $medicineType) {
        $medicineType->delete();
        return response()->noContent();
    }


    //========= Medicine Type Count
    public function MedicineTypeCount()
    {
        try {
            $companyId = getCompanyId();
            $count = MedicineType::where('company_id', $companyId)->count();
            return response()->json(['count' => $count]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Unable to fetch company user count'], 500);
        }
    }
}
