<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Purchase;
use App\Models\Product;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Js;
use Inertia\Inertia;


class PurchaseController extends Controller
{
    public function index()
    {
        $companyId = getCompanyId();
        // Fetch purchases where the related product and supplier belong to the company
        $purchases = Purchase::with(['product:id,name,company_id', 'supplier:id,name,company_id'])
            ->whereHas('product', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
            ->whereHas('supplier', function ($query) use ($companyId) {
                $query->where('company_id', $companyId);
            })
            ->orderBy('created_at', 'desc')
            ->get();
    
        return response()->json($purchases);
    }
    
    
    public function store(Request $request)
    {
        $companyId = getCompanyId();

        // Validate product_id, supplier_id, quantity, and date only
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'supplier_id' => 'required|exists:suppliers,id',
            'quantity' => 'required|numeric|min:1',
            'purchase_date' => 'required|date',
        ]);

        try {
            // Fetch the product to get the purchase_price
            $product = Product::findOrFail($validated['product_id']);
            $purchasePrice = $product->purchase_price;

            // Total cost = unit purchase price × quantity
            $totalMoney = $validated['quantity'] * $purchasePrice;

            // Create a new purchase record
            $purchase = Purchase::create([
                'product_id' => $validated['product_id'],
                'supplier_id' => $validated['supplier_id'],
                'company_id' => $companyId,
                'quantity' => $validated['quantity'],
                'purchase_price' => $purchasePrice, // use from Product
                'purchase_date' => $validated['purchase_date'],
            ]);

            // Update the supplier's total money
            $supplier = Supplier::find($validated['supplier_id']);
            if ($supplier) {
                $supplier->increment('total_money', $totalMoney);
            }

            return response()->json(['success' => true, 'data' => $purchase]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()]);
        }
    }
    
    public function show(Purchase $request)
    {

        // Render a new page using Inertia and pass the products to it
        
    }

public function update(Request $request, $id)
{
    Log::info('Welcome to update controller in purchase');

    $companyId = getCompanyId();
    $purchase = Purchase::find($id);

    if (!$purchase) {
        return response()->json(['message' => 'purchase not found'], 404);
    }

    // Validate input (no purchase_price in request)
    $validatedData = $request->validate([
        'product_id' => 'required|exists:products,id',
        'supplier_id' => 'required|exists:suppliers,id',
        'quantity' => 'required|numeric|min:1',
        'purchase_date' => 'required|date',
    ]);

    // Fetch the product's purchase price
    $product = Product::findOrFail($validatedData['product_id']);
    $purchasePrice = $product->purchase_price;

    // Calculate total money before and after update
    $oldTotalMoney = $purchase->quantity * $purchase->purchase_price;
    $newTotalMoney = $validatedData['quantity'] * $purchasePrice;

    // Update supplier's total money
    $supplier = Supplier::find($purchase->supplier_id);
    if ($supplier) {
        $supplier->total_money = $supplier->total_money - $oldTotalMoney + $newTotalMoney;
        $supplier->save();
    }

    // Apply update to purchase
    $purchase->update([
        'product_id' => $validatedData['product_id'],
        'supplier_id' => $validatedData['supplier_id'],
        'company_id' => $companyId,
        'quantity' => $validatedData['quantity'],
        'purchase_price' => $purchasePrice, // From Product model
        'purchase_date' => $validatedData['purchase_date'],
    ]);

    return response()->json([
        'message' => 'Purchase updated successfully',
        'purchase' => $purchase
    ], 200);
}


    public function destroy($id)
    {
        try {
            // Find the supp$supplier by ID or fail
            $purchase = Purchase::findOrFail($id);
    
            // Delete the supp$purchase record from the database
            $purchase->delete();
    
            return response()->json(['message' => 'purchase and its image deleted successfully.'], 200);
        } catch (\Exception $e) {
            // Log any errors that occur
            return response()->json(['error' => 'Internal Server Error'], 500);
        }
    }


    // In PurchaseController.php
    public function Checkout(Request $request) 
    { 
        $products = $request->input('products');

        if (!$products || count($products) === 0){
            return redirect()->back()->with('error', 'No products selected for checkout');
        }

        // Log::info('product want to be rendered');

        return response()->json([ 'success' => true, 'products' => $products, ]);

    }

    public function showCheck(Request $request)

    {
        $products = $request->input('products');

        // Log::info('Checkout Products:', ['products' => $products]);

        // $products = $products ? json_decode($products, true) : [];

        if (!$products || count($products) === 0){
            return redirect()->back()->with('error', 'No products selected for checkout');
        }

        // Log::info('Checkout Products2:', ['products' => $products]);

        return Inertia::render('Company/Sales/CheckoutSummary', [
            'products' => $products,
        ]);

    }

    //==========================


    public function countPurchasedProducts()
    {
        try {
            $companyId = null;

            // Determine if the request is from an admin or a user
            $companyId = getCompanyId();
            // Calculate the count of all products
            $purchasedCount = Purchase::where('company_id', $companyId)->count();
 
            return response()->json(['purchased_products_count' => $purchasedCount], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Internal Server Error'], 500);
        }
    }
   

}
