<?php

namespace App\Http\Controllers;
use App\Models\Stock;
use App\Models\ReversalLog;
use App\Models\SaleItem;
use App\Models\Sale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ReversalController extends Controller
{
    public function store(Request $request)
    {
        $companyId = getCompanyId();

        $request->validate([
            'product_id' => 'required|exists:sale_items,id',
            'reason' => 'required|string|max:255',
        ]);

        // Fetch the sale item
        $saleItem = SaleItem::findOrFail($request->product_id);

        // Log::info(['sale item:'=> $saleItem]);

        // Create a reversal log
        $reversalLog = ReversalLog::create([
            'company_id' => $saleItem->company_id,
            'user_id' => auth('user')->id(),
            'product_name' => $saleItem->product_name,
            'sales_price' => $saleItem->sales_price,
            'quantity' => $saleItem->quantity,
            'total_price' => $saleItem->total_price,
            'profit' => $saleItem->profit,
            'sale_date' => $saleItem->sale_date,
            'reason' => $request->reason,
            'status' => 'pending',
            'reverse_date' => Carbon::now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Reversal request submitted successfully.',
            'data' => $reversalLog,
        ]);
    }



    public function showPending(){
        $companyId = getCompanyId();
        $sales = ReversalLog::where('company_id', $companyId)
        ->orderBy('created_at', 'desc')
        ->where('status', 'pending')
        ->get();
        return response()->json($sales);
    }




    public function approveReversal($id)
    {
        $reversalLog = ReversalLog::findOrFail($id);
        $reversalLog->update(['status' => 'approved']);
        $stock = Stock::where('product_name', $reversalLog->product_name)
            ->where('company_id', $reversalLog->company_id)
            ->first();
    
        if ($stock) {
            $stock->quantity += $reversalLog->quantity;
        } else {
            $stock = new Stock([
                'company_id' => $reversalLog->company_id,
                'product_name' => $reversalLog->product_name,
                'sales_price' => $reversalLog->sales_price,
                'quantity' => $reversalLog->quantity,
            ]);
        }
    
        if ($stock->quantity == 0) {
            $stock->status = 'Out of Stock';
        } elseif ($stock->quantity < 10) {
            $stock->status = 'Low Stock';
        } else {
            $stock->status = 'High Stock';
        }
    
        $stock->save();

        SaleItem::where('product_name', $reversalLog->product_name)
            ->where('sale_date', $reversalLog->sale_date)
            ->delete();


        $sales = Sale::where('created_at', $reversalLog->sale_date)
        ->where('company_id', $reversalLog->company_id)
        ->get();



        Log::info(['sale products:'=> $sales]);

        
        foreach ($sales as $sale){
            $products = json_decode($sale->products, true);

            Log::info(['sale products:'=> $products]);

            $updatedProducts = array_filter($products, function ($product) use ($reversalLog){
                return $product['product_name'] !== $reversalLog->product_name;
            });

            Log::info(['sale products:'=> $updatedProducts]);

            if(empty($updatedProducts)){
                $sale->delete();
            }else{
                $sale->update(['products'=>json_encode(array_values($updatedProducts))]);
            }
        }
    
        return response()->json([
            'success' => true,
            'message' => 'Reversal approved successfully.',
        ]);
    }


    public function reversalLogs(){
        $companyId = getCompanyId();
        $sales = ReversalLog::where('company_id', $companyId)
        ->orderBy('created_at', 'desc')
        ->where('status', 'approved')
        ->get();
        return response()->json($sales);
    }


    public function allReversalLogs(){
        $companyId = getCompanyId();
        $sales = ReversalLog::where('company_id', $companyId)
        ->orderBy('created_at', 'desc')
        ->get();
        return response()->json($sales);
    }
}
