<?php

namespace App\Http\Controllers;

use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\SalePayment;
use App\Models\Product;
use App\Models\Stock;
use App\Models\CompanyProfile;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inertia\Inertia;

class SaleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $companyId = getCompanyId();
        $validated = $request->validate([
            'customerName' => 'required|string|max:255',
            'phoneNumber' => 'required|string|max:255',
            'customerType' => 'required|string|max:255',
            'email' => 'nullable|string|required_if:customerType,partial',
            'address' => 'nullable|string|required_if:customerType,partial',
            'saleDate' => 'required|date',
            'amountPaid' => 'required|numeric|min:0',
            'totalAmount' => 'required|numeric|min:0',
            'products' => 'required|array',
        ]);


        $partialCustomerId = null;

         // Handle partial customer
    if ($request->customerType === 'partial') {
        $existings = Sale::where('phone_number', $request->phoneNumber)
                            ->where('customer_type', 'partial')
                            ->where('customer_name', $request->customerName)
                            ->where('email', $request->email)
                            ->where('company_id', $companyId)
                            ->first();

        $partialCustomerId = $existings?->partial_customer_id ?? Str::uuid()->toString();
    }

        $sale = Sale::create([
            'customer_name' => $request->customerName,
            'phone_number' => $request->phoneNumber,
            'customer_type' => $request->customerType,
            'email' => $request->email ?? null,
            'address' => $request->address ?? null,
            'sale_date' => $request->saleDate,
            'amount_paid' => $request->amountPaid,
            'total_amount' => $request->totalAmount,
            'products' => json_encode($request->products),
            'company_id'=> $companyId,
            'partial_customer_id' => $partialCustomerId,
        ]);


        SalePayment::create([
            'company_id' => $companyId,
            'sale_id' => $sale->id,
            'total_amount' => $request->totalAmount,
            'paid_amount' => $request->amountPaid,
            'paid_at' => $request->saleDate,
        ]);

        // Log::info(['sales:'=>$sale]);

        $totalPaid = $request->amountPaid;
        $totalAmount = $request->totalAmount;
    
        foreach ($request->products as $product) {
            $productTotal = $product['sales_price'] * $product['quantity'];

            $productData = Stock::find($product['product_id']);
            $purchasePrice = Product::findOrFail($product['product_id']);
            $pur_price = $purchasePrice->purchase_price;

            $paidForThisItem = min($productTotal, $totalPaid);
            $isFullyPaid = $paidForThisItem >= $productTotal;

            SaleItem::create([
                'company_id' => $companyId,
                'sale_id' => $sale->id,
                'product_name' => $product['product_name'],
                'sales_price' => $product['sales_price'],
                'quantity' => $product['quantity'],
                'total_price' => $productTotal,
                'paid_amount' => $paidForThisItem,
                'is_fully_paid' => $isFullyPaid,
                'profit' => $paidForThisItem - ($pur_price * ($paidForThisItem / $product['sales_price'])),
                'sale_date' => Carbon::now(),
            ]);
    
            $totalPaid -= $paidForThisItem;
        }
    
        // Laravel example
        $lastInvoice = SaleItem::orderBy('id', 'desc')->first();
        $nextId = $lastInvoice ? $lastInvoice->id + 1 : 1;
        $invoiceNumber = 'INV-' . date('Y') . '-' . str_pad($nextId, 6, '0', STR_PAD_LEFT);

        return Inertia::location(route('bill', [
            'customerName' => $sale->customer_name,
            'phoneNumber' => $sale->phone_number,
            'customerType' => $sale->customer_type,
            'email' => $sale->email,
            'address' => $sale->address,
            'saleDate' => $sale->sale_date,
            'amountPaid' => $sale->amount_paid,
            'totalAmount' => $sale->total_amount,
            'products' => json_decode($sale->products),
            'company_id' => $sale->company_id,
            'invoiceNumber' => $invoiceNumber,
        ]));
    }

    //============ Print Bill ===========

    public function PrintBill(Request $request)
    {
        $companyDetails = CompanyProfile::where('company_id', $request->input('company_id'))->first();

        $response = response()->json([
            'customerName' => $request->input('customerName'),
            'saleDate' => $request->input('saleDate'),
            'amountPaid' => $request->input('amountPaid'),
            'products' => $request->input('products', []),
        ]);

        return Inertia::render('Company/Sales/Bill', [
            'response' => $response,
            'companyDetails' => $companyDetails, // ✅ Add this
        ]);
    }

    //============================================================

    public function PartialCustomers(){
        $companyId = getCompanyId();
        $customers = Sale::where('customer_type', 'partial')
                ->where('company_id', $companyId)
                ->whereColumn('amount_paid', '<', 'total_amount')
                ->orderBy('created_at', 'desc')
                ->get();

        foreach ($customers as $customer){
            $customer->products = json_decode($customer->products);
        }

        return response()->json($customers);
    }

    //================ report customers partial

    
    public function PartialCustomersList()
    {
        $companyId = getCompanyId();
    
        $allSales = Sale::where('customer_type', 'partial')
            ->where('company_id', $companyId)
            ->whereNotNull('partial_customer_id')
            ->orderBy('created_at', 'desc')
            ->get();
    
        // Group by 'partial_customer_id' and calculate the sum of 'total_amount', 'amount_paid', and 'remain'
        $customers = $allSales
            ->groupBy('partial_customer_id')
            ->map(function ($sales, $partialCustomerId) {
                $totalAmount = $sales->sum('total_amount');
                $amountPaid = $sales->sum('amount_paid');
                $remain = $totalAmount - $amountPaid;
    
                // Get the first sale's customer details (assuming all sales have the same customer details)
                $sale = $sales->first();
    
                return [
                    'id' => $partialCustomerId,
                    'customer_name' => $sale->customer_name,
                    'phone_number' => $sale->phone_number,
                    'email' => $sale->email,
                    'address' => $sale->address,
                    'total_amount' => $totalAmount,
                    'amount_paid' => $amountPaid,
                    'remain' => $remain,
                ];
            })
            ->values();
    
        return response()->json($customers);
    }



    //=======================

    public function getProductsByPartialCustomerId($id)
    {
        $sales = Sale::where('partial_customer_id', $id)
            ->get();
        $allProducts = [];
        foreach ($sales as $sale) {
            $products = json_decode($sale->products, true); // assuming 'products' is a JSON column
            if (is_array($products)) {
                $allProducts = array_merge($allProducts, $products);
            }
        }

        return response()->json($allProducts);
    }


    //=================== Payments list
    public function getCustomerPayments($customerName)
    {
        // Get all sales by customer name
        $sales = Sale::with('payments')->where('customer_name', $customerName)->get();

        // Structure the response
        $payments = [];

        foreach ($sales as $sale) {
            foreach ($sale->payments as $payment) {
                $payments[] = [
                    'payment_number' => $payment->id,
                    'sale_id' => $sale->id,
                    'total_amount' => $payment->total_amount,
                    'paid_amount' => $payment->paid_amount,
                    'paid_at' => $payment->paid_at,
                    'customer_name' => $sale->customer_name,
                    'phone_number' => $sale->phone_number,
                    'email' => $sale->email,
                ];
            }
        }

        return response()->json($payments);
    }


    //========================= walk-in

    public function WalkInCustomersList(){
        $companyId = getCompanyId();
        $customers = Sale::where('customer_type', 'walk-in')
                ->where('company_id', $companyId)
                ->orderBy('created_at', 'desc')
                ->get();
        
        foreach ($customers as $customer){
            $customer->products = json_decode($customer->products);
        }
        return response()->json($customers);
    }



    public function updatePayment(Request $request, $id)
    {
        $request->validate([
            'amount_paid' => 'required|numeric|min:0.01',
        ]);

        DB::transaction(function () use ($request, $id) {
            $sale = Sale::findOrFail($id);

            $newPaymentAmount = $request->amount_paid;
            $totalAlreadyPaid = $sale->amount_paid;
            $newTotal = $totalAlreadyPaid + $newPaymentAmount;

            if ($newTotal > $sale->total_amount) {
                throw new \Exception('Payment exceeds total sale amount.');
            }

            // Record payment in sale_payments table
            SalePayment::create([
                'company_id' => $sale->company_id,
                'sale_id' => $sale->id,
                'total_amount' => $sale->total_amount,
                'paid_amount' => $newPaymentAmount,
                'paid_at' => now(),
            ]);

            // Update total paid on the sale
            $sale->amount_paid = $newTotal;
            $sale->save();

            $amountToApply = $newPaymentAmount;

            // Get unpaid sale items in order
            $saleItems = SaleItem::where('sale_id', $sale->id)
                ->where('is_fully_paid', false)
                ->orderBy('id')
                ->get();

            foreach ($saleItems as $item) {
                if ($amountToApply <= 0) break;

                $remainingForItem = $item->total_price - $item->paid_amount;
                $paymentForItem = min($remainingForItem, $amountToApply);

                // Update paid amount
                $item->paid_amount += $paymentForItem;

                // Mark as fully paid if applicable
                if ($item->paid_amount >= $item->total_price) {
                    $item->is_fully_paid = true;
                }

                // Get purchase price from product
                // You should ideally link SaleItem with product_id instead of relying on name
                $product = Product::where('name', $item->product_name)->first();
                if ($product) {
                    $unitPurchasePrice = $product->purchase_price;
                    $unitSalesPrice = $item->sales_price;

                    $quantityPortion = $paymentForItem / $unitSalesPrice;
                    $additionalProfit = $paymentForItem - ($unitPurchasePrice * $quantityPortion);

                    $item->profit += $additionalProfit;
                }

                $item->save();
                $amountToApply -= $paymentForItem;
            }
        });

        return response()->json(['message' => 'Payment updated and recorded successfully.']);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }



    //@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
    //@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@2
    //@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

    //@@@@@@@@@@@@@@@@@@@@@@@@@     Reports @@@@@@@@@@@@@@@@@@@@@@@@@@@
    

        ////////////////////////////     Top Five Best Selling Products /////////////////

        public function TopFiveSales(){

            $companyId = getCompanyId();
            $fiveSales = SaleItem::where('company_id', $companyId)
            ->select('product_name',
                    DB::raw('SUM(quantity) as total_quantity'),
                    DB::raw('SUM(total_price) as total_sales'),
                    DB::raw('SUM(profit) as total_profit'))
            ->groupBy('product_name')
            ->orderByDesc('total_quantity')
            ->limit(5)
            ->get();

            // Log::info(['five best'=> $fiveSales]);

            return response()->json(['message'=>$fiveSales]);
        }


        //////////////////////// Recent Sales /////////////////


        public function getRecentSales(){

            $companyId = getCompanyId();
            $recentSales = SaleItem::where('company_id', $companyId)  
            ->orderByDesc('created_at')
            ->limit(10)
            ->get();

            return response()->json(['message'=>$recentSales]);
        }

        //////////////////////// Total Current Month Sales ///////////////////////

        public function CurrentMonthSales(){
            $companyId = getCompanyId();
            $currentMonth = Carbon::now()->month;
            $currentYear = Carbon::now()->year;

            $sales = SaleItem::where('company_id', $companyId)
                ->whereMonth('sale_date', $currentMonth)
                ->whereYear('sale_date', $currentYear)
                ->select(
                    DB::raw('SUM(quantity) as total_quantity'),
                    DB::raw('SUM(total_price) as total_sales'),
                    DB::raw('COUNT(id) as total_transactions'),
                    DB::raw('SUM(profit) as total_profit'),
                )->first();

            // Log::info(['five best'=> $fiveSales]);

            return response()->json(['message'=>$sales]);
        }

        /////////////////// Total All Sales in one row ////////////


        public function allSales(){
            $companyId = getCompanyId();

            $sales = SaleItem::where('company_id', $companyId)
                ->select(
                    DB::raw('SUM(quantity) as total_quantity'),
                    DB::raw('SUM(total_price) as total_sales'),
                    DB::raw('COUNT(id) as total_transactions'),
                    DB::raw('SUM(profit) as total_profit'),
                )->first();
            // Log::info(['five best'=> $fiveSales]);

            return response()->json(['message'=>$sales]);
        }

///////////////////////// List of all sales ///////////////////


        public function allProductSales(){

            $companyId = getCompanyId();
            $sales = SaleItem::where('company_id', $companyId)
            ->orderBy('created_at', 'desc')
            ->get();

            return response()->json(['message'=>$sales]);
        }


/////////////////// List of all current sales ////////////////////////////

        public function allCurrentMonthSales(){
            $companyId = getCompanyId();
            $currentMonth = Carbon::now()->month;
            $currentYear = Carbon::now()->year;

            $sales = SaleItem::where('company_id', $companyId)
                ->whereMonth('sale_date', $currentMonth)
                ->whereYear('sale_date', $currentYear)
                ->get();

            return response()->json(['message'=>$sales]);
        }

        /////////////  Get Chart data of month sales 

        public function getChartMonthSales(Request $request)
        {
            $companyId = getCompanyId();
            $year = $request->input('year', date('Y'));
    
            // Query to sum sales amounts grouped by month
            $sales = SaleItem::where('company_id', $companyId)
                ->select(
                    DB::raw('MONTH(sale_date) as month'),
                    DB::raw('SUM(total_price) as total_sales')
                )
                ->whereYear('sale_date', $year)
                ->groupBy(DB::raw('MONTH(sale_date)'))
                ->orderBy(DB::raw('MONTH(sale_date)'), 'asc')
                ->get();
    
            // Initialize an array with zeros for all 12 months
            $monthlySales = array_fill(1, 12, 0);
    
            // Populate the array with actual sales data
            foreach ($sales as $sale) {
                $monthlySales[(int)$sale->month] = (float)$sale->total_sales;
            }
    
            // Month labels for the chart
            $labels = [
                'January', 'February', 'March', 'April', 'May', 'June',
                'July', 'August', 'September', 'October', 'November', 'December'
            ];
    
            // Return the data as JSON
            return response()->json([
                'labels' => $labels,
                'data'   => array_values($monthlySales),
                'year'   => $year
            ]);
        }


        //========= Today sales ==============

        public function TodaySales(){
            $companyId = getCompanyId();

            $sales = SaleItem::where('company_id', $companyId)
                ->whereDate('sale_date', Carbon::now())
                ->get();

            return response()->json(['message'=>$sales]);
        }

        //============ Total Sales of Today ======

        public function TotalTodaySales(){
            $companyId = getCompanyId();

            $sales = SaleItem::where('company_id', $companyId)
                ->whereDate('sale_date', Carbon::now())
                ->select(
                    DB::raw('SUM(quantity) as total_quantity'),
                    DB::raw('SUM(total_price) as total_sales'),
                    DB::raw('COUNT(id) as total_transactions'),
                    DB::raw('SUM(profit) as total_profit'),
                )->first();

            // Log::info(['Today:'=> $sales]);

            return response()->json(['message'=>$sales]);
        }

        ///======================================  get today chart sales

        public function getCircleLineTodaySales(Request $request)
        {
            $companyId = getCompanyId();
            $date = Carbon::now()->toDateString();
    
            // Query to sum sales amounts grouped by month
            $sales = SaleItem::where('company_id', $companyId)
                ->select(
                    DB::raw('HOUR(sale_date) as hour'),
                    DB::raw('SUM(total_price) as total_sales')
                )
                ->wheredate('sale_date', $date)
                ->groupBy(DB::raw('HOUR(sale_date)'))
                ->orderBy(DB::raw('HOUR(sale_date)'), 'asc')
                ->get();
    
            // Initialize an array with zeros for all 12 months
            $hourlySales = array_fill(0, 24, 0);
    
            // Populate the array with actual sales data
            foreach ($sales as $sale) {
                $hourlySales[(int)$sale->hour] = (float)$sale->total_sales;
            }
    
            // Month labels for the chart
            $labels = array_map(fn($hour)=> sprintf('%02d:00', $hour), range(0, 23));
    
            // Return the data as JSON
            return response()->json([
                'labels' => $labels,
                'data'   => array_values($hourlySales),
                'date'   => $date
            ]);
        }


        //=================

        public function getBarchartTodaySales(Request $request)
        {
            $companyId = getCompanyId();
        
            // Query with timezone conversion
            $sales = SaleItem::where('company_id', $companyId)
                ->whereDay('sale_date', Carbon::today())
                ->selectRaw("HOUR(sale_date) as sale_hour, SUM(total_price) as total_sales")
                ->groupBy('sale_hour')
                ->orderBy('sale_hour', 'asc')
                ->get();
        
            return response()->json([
                'sales' => $sales
            ]);
        }
}
