<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Company;
use App\Models\CompanyUser;
use App\Models\SystemSetting;
use App\Models\CompanyProfile;
use Illuminate\Support\Facades\DB;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\Product;
use App\Models\Purchase;
use App\Models\Stock;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Inertia\Inertia;

class SystemAdminDashboardController extends Controller
{
    public function Reports(Request $request)
    {

        $dateRange = $request->input('date_range', 'today');
        $companyId = $request->input('company_id');
       
        // Parse date range
        switch ($dateRange) {
            case 'today':
                $startDate = Carbon::today();
                $endDate = Carbon::today()->endOfDay();
                break;
            case 'yesterday':
                $startDate = Carbon::yesterday();
                $endDate = Carbon::yesterday()->endOfDay();
                break;
            case 'this_week':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                break;
            case 'last_week':
                $startDate = Carbon::now()->subWeek()->startOfWeek();
                $endDate = Carbon::now()->subWeek()->endOfWeek();
                break;
            case 'this_month':
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                break;
            case 'last_month':
                $startDate = Carbon::now()->subMonth()->startOfMonth();
                $endDate = Carbon::now()->subMonth()->endOfMonth();
                break;
            case 'this_year':
                $startDate = Carbon::now()->startOfYear();
                $endDate = Carbon::now()->endOfYear();
                break;
            case 'custom':
                $startDate = Carbon::parse($request->input('start_date'));
                $endDate = Carbon::parse($request->input('end_date'));
                break;
            default:
                $startDate = Carbon::today();
                $endDate = Carbon::today()->endOfDay();
        }

        // Get all companies for filter dropdown
        $companies = Company::all();

        // Sales Report
        $salesQuery = SaleItem::whereBetween('sale_date', [$startDate, $endDate]);
        if ($companyId) {
            $salesQuery->where('company_id', $companyId);
        }
        $salesReport = $salesQuery->get();

        // Sales Summary
        $salesSummary = [
            'total_sales' => $salesReport->sum('total_price'),
            'total_profit' => $salesReport->sum('profit'),
            'total_items_sold' => $salesReport->sum('quantity'),
            'average_sale' => $salesReport->avg('total_price'),
        ];

        // Sales by Company
        $salesByCompany = $salesReport->groupBy('company_id')->map(function ($items) {
            return [
                'total_sales' => $items->sum('total_price'),
                'total_profit' => $items->sum('profit'),
                'count' => $items->count(),
            ];
        });

        // Stock Status Report
        $stockQuery = Stock::query();
        if ($companyId) {
            $stockQuery->where('company_id', $companyId);
        }
        $stockReport = $stockQuery->get();

        $stockStatusSummary = $stockReport->groupBy('status')->map->count();

        // Low Stock Items
        $lowStockItems = $stockQuery->clone()
            ->where('status', 'Low Stock')
            ->orderBy('quantity', 'asc')
            ->limit(10)
            ->get();

        // Out of Stock Items
        $outOfStockItemsQuery = Product::leftJoin('stocks', 'products.id', '=', 'stocks.product_id');
        if ($companyId) {
            $outOfStockItemsQuery->where('products.company_id', $companyId); // Specify table name
        }
        $outOfStockItems = $outOfStockItemsQuery
            ->where(function($query) {
                $query->whereNull('stocks.id') // No stock record
                    ->orWhere('stocks.quantity', '<=', 0); // Or zero quantity
            })
            ->select('products.*') // Only select product columns
            ->with(['company', 'stock']) // Eager load
            ->orderBy('products.name')
            ->limit(10)
            ->get();


        
        // Purchases Report
        $purchasesQuery = Purchase::with('product')->whereBetween('purchase_date', [$startDate, $endDate]);
        if ($companyId) {
            $purchasesQuery->where('company_id', $companyId);
        }
        $purchasesReport = $purchasesQuery->get();

        // Top Selling Products
        $topSellingProducts = $salesReport->groupBy('product_name')->map(function ($items) {
            return [
                'total_sold' => $items->sum('quantity'),
                'total_revenue' => $items->sum('total_price'),
            ];
        })->sortByDesc('total_sold')->take(10);

        return Inertia::render('SystemAdmin/Reports/Index', [
            'salesSummary' => $salesSummary,
            'salesByCompany' => $salesByCompany,
            'stockStatusSummary' => $stockStatusSummary,
            'lowStockItems' => $lowStockItems,
            'outOfStockItems' => $outOfStockItems,
            'purchasesReport' => $purchasesReport,
            'topSellingProducts' => $topSellingProducts,
            'companies' => $companies,
            'filters' => [
                'date_range' => $dateRange,
                'company_id' => $companyId,
                'start_date' => $request->input('start_date'),
                'end_date' => $request->input('end_date'),
            ],
            'dateRange' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d'),
            ],
            'userRole' => auth()->user()->role,
        ]);
    }

    //===============================  Top company methods

    public function TopCompanyReports(Request $request)
    {
        // Get filter parameters
        $timeRange = $request->input('time_range', 'monthly');
        $metric = $request->input('metric', 'revenue');
        $limit = $request->input('limit', 10);

        // Base query for companies
        $query = Company::with(['profile', 'users'])
            ->withCount('users');

        // Only add date filtering if not "all_time"
        if ($timeRange !== 'all_time') {
            $dateRange = $this->getDateRange($timeRange);
        
            if ($metric === 'revenue') {
                $query->addSelect([
                    'total_revenue' => SaleItem::select(DB::raw('COALESCE(SUM(total_price), 0)'))
                        ->whereColumn('company_id', 'companies.id')
                        ->whereBetween('sale_date', $dateRange)
                ]);
            } elseif ($metric === 'products') {
                $query->addSelect([
                    'total_products' => DB::table('products')
                        ->select(DB::raw('COUNT(*)'))
                        ->whereColumn('company_id', 'companies.id')
                        ->whereBetween('created_at', $dateRange)
                ]);
            } elseif ($metric === 'sales') {
                $query->addSelect([
                    'total_sales' => SaleItem::select(DB::raw('COUNT(*)'))
                        ->whereColumn('company_id', 'companies.id')
                        ->whereBetween('sale_date', $dateRange)
                ]);
            }
        } else {
            // For "all_time", we need to add these columns without date filtering
            if ($metric === 'revenue') {
                $query->addSelect([
                    'total_revenue' => SaleItem::select(DB::raw('COALESCE(SUM(total_price), 0)'))
                        ->whereColumn('company_id', 'companies.id')
                ]);
            } elseif ($metric === 'products') {
                $query->addSelect([
                    'total_products' => DB::table('products')
                        ->select(DB::raw('COUNT(*)'))
                        ->whereColumn('company_id', 'companies.id')
                ]);
            } elseif ($metric === 'sales') {
                $query->addSelect([
                    'total_sales' => SaleItem::select(DB::raw('COUNT(*)'))
                        ->whereColumn('company_id', 'companies.id')
                ]);
            }
        }

        // Order by selected metric
        switch ($metric) {
            case 'revenue':
                $query->orderByDesc('total_revenue');
                break;
            case 'products':
                $query->orderByDesc('total_products');
                break;
            case 'sales':
                $query->orderByDesc('total_sales');
                break;
            case 'users':
                $query->orderByDesc('users_count');
                break;
            default:
                $query->orderByDesc('total_revenue');
        }

        // Get top companies
        $companies = $query->take($limit)->get();

        // Define available filter options
        $metrics = [
            ['value' => 'revenue', 'label' => 'Revenue', 'icon' => 'ri-money-dollar-circle-line'],
            ['value' => 'sales', 'label' => 'Sales Count', 'icon' => 'ri-shopping-bag-line'],
            ['value' => 'products', 'label' => 'Products', 'icon' => 'ri-box-line'],
            ['value' => 'users', 'label' => 'Users', 'icon' => 'ri-user-line']
        ];

        $timeRanges = [
            ['value' => 'daily', 'label' => 'Today'],
            ['value' => 'weekly', 'label' => 'This Week'],
            ['value' => 'monthly', 'label' => 'This Month'],
            ['value' => 'yearly', 'label' => 'This Year'],
            ['value' => 'all_time', 'label' => 'All Time']
        ];

        return Inertia::render('SystemAdmin/Reports/TopCompanies', [
            'companies' => $companies,
            'filters' => [
                'time_range' => $timeRange,
                'metric' => $metric,
                'limit' => $limit
            ],
            'metrics' => $metrics,
            'timeRanges' => $timeRanges,
            'limits' => [5, 10, 15, 20],
            'userRole' => auth()->user()->role,
        ]);
    }

    private function getDateRange($timeRange)
    {
        switch ($timeRange) {
            case 'daily':
                return [now()->startOfDay(), now()->endOfDay()];
            case 'weekly':
                return [now()->startOfWeek(), now()->endOfWeek()];
            case 'monthly':
                return [now()->startOfMonth(), now()->endOfMonth()];
            case 'yearly':
                return [now()->startOfYear(), now()->endOfYear()];
            case 'all_time':
                return null;
            default:
                return [now()->startOfMonth(), now()->endOfMonth()];
        }
    }

    //=======================================================================


    public function CompanyCount()
    {
        try {
            $count = Company::all()->count();
            return response()->json(['count' => $count]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Unable to fetch company count'], 500);
        }
    }

        //=================

        public function AllCompanyUserCount()
        {
            try {
                $count = CompanyUser::all()->count();
                return response()->json(['count' => $count]);
            } catch (\Exception $e) {
                return response()->json(['error' => 'Unable to fetch company user count'], 500);
            }
        }


                //============ Total Sales of Today ======

            public function TotalTransactions(){
        
                try {
                    $count = SaleItem::all()->count();
                    return response()->json(['count' => $count]);
                } catch (\Exception $e) {
                    return response()->json(['error' => 'Unable to fetch company user count'], 500);
                }
            }


            //=============== Expired products

            public function getAllExpiredProducts(){
                try {
                    $count = Product::where('expiration_date', '<', Carbon::now())
                        ->count();
                        return response()->json(['count' => $count]);
                } catch (\Exception $e) {
                    return response()->json(['error' => 'Unable to fetch company expired product count'], 500);
                }
            }


            //============== Active user count
            
            public function getSystemActiveUserCount()
            {
                $activeUsers = CompanyUser::where('last_seen', '>=' , Carbon::now()->subMinutes(5))->count();
                return response()->json(['active_users' => $activeUsers]);
            }


            //============= Chart Month Sales
            public function getSystemChartMonthSales(Request $request)
            {
                $year = $request->input('year', date('Y'));
        
                // Query to sum sales amounts grouped by month
                $sales = SaleItem::select(
                        DB::raw('MONTH(sale_date) as month'),
                        DB::raw('SUM(total_price) as total_sales')
                    )
                    ->whereYear('sale_date', $year)
                    ->groupBy(DB::raw('MONTH(sale_date)'))
                    ->orderBy(DB::raw('MONTH(sale_date)'), 'asc')
                    ->get();
        
                // Initialize an array with zeros for all 12 months
                $monthlySales = array_fill(1, 12, 0);
        
                // Populate the array with actual sales data
                foreach ($sales as $sale) {
                    $monthlySales[(int)$sale->month] = (float)$sale->total_sales;
                }
        
                // Month labels for the chart
                $labels = [
                    'January', 'February', 'March', 'April', 'May', 'June',
                    'July', 'August', 'September', 'October', 'November', 'December'
                ];
        
                // Return the data as JSON
                return response()->json([
                    'labels' => $labels,
                    'data'   => array_values($monthlySales),
                    'year'   => $year
                ]);
            }



    //============== Company Profiles
    public function showCompanyProfiles()
    {
        try {
            $profiles = CompanyProfile::all();
            return Inertia::render('SystemAdmin/CompanyProfiles', [
                'profiles' => $profiles,
                'userRole' => auth()->user()->role,
            ]);
        } catch (\Exception $e) {
            return Inertia::render('SystemAdmin/CompanyProfiles', [
                'profiles' => [],
                'error' => $e->getMessage(),
            ]);
        }
    }

    //============== Company Users
    public function getCompanyUsers()
    {
        try {
            $companies = Company::all(); // For dropdown
            $users = CompanyUser::with('company')->get();

            return Inertia::render('SystemAdmin/CompanyUsers', [
                'companyUsers' => $users,
                'companies' => $companies,
                'userRole' => auth()->user()->role,
            ]);
        } catch (\Exception $e) {
            return Inertia::render('SystemAdmin/CompanyUsers', [
                'companyUsers' => [],
                'companies' => [],
                'error' => $e->getMessage(),
                'userRole' => auth()->user()->role,
            ]);
        }
    }



    //================== System Backup Manual

    public function getSystemCompanies()
    {
    
        $database = env('DB_DATABASE');
        $user = env('DB_USERNAME', 'default_user');
        $password = env('DB_PASSWORD', 'default_password');
    
        $timestamp = Carbon::now()->format('Y-m-d_H-i-s');
        $folderPath = "C:/backup/SystemAdminBackupData/";
        $filename = "Backup-{$timestamp}.sql";
        $storagePath = $folderPath . $filename;
    
        if (!is_dir($folderPath)) {
            mkdir($folderPath, 0777, true);
        }
    
        // Tables to back up
        $tables = [
            'users','companies',
            'company_users', 'categories', 'company_profiles',
            'products', 'suppliers', 'purchases', 'stocks',
            'sales', 'sale_items','payments', 
            'system_settings', 'medicine_types', 'expenses',
            'reversal_logs', 'sale_payments',
        ];
    
        // Initialize backup file
        file_put_contents($storagePath, ""); // Clear or create the backup file
    
        foreach ($tables as $table) {
            $command = "mysqldump -u {$user} --password=\"{$password}\" {$database} {$table} >> \"{$storagePath}\"";
            exec($command, $output, $resultCode);
    
            // Log::info(['result code backup:'=> $resultCode]);

            if ($resultCode !== 0) {
                return response()->json(['message' => "Backup failed for table {$table}"], 500);
            }
        }
    
        return response()->json(['message' => 'Backup completed successfully'], 200);
    }
}


